import argparse
import builtins
import os
import random
import time
import warnings
import timm

import torch
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.multiprocessing as mp
import torch.nn as nn
from braincog.model_zoo.sew_resnet import *
from braincog.model_zoo.vgg_snn import *
from braincog.base.node import *
from moco.loader import *
from moco.builder import *
from moco.utils import *
import wandb
from torch.amp import autocast, GradScaler
from functools import partial
parser = argparse.ArgumentParser(description="PyTorch ImageNet Training")

parser.add_argument(
    "--model_config",
    default=None,
    help="model config file",
)
parser.add_argument(
    "--dataset_config",
    default=None,
    help="dataset config file",
)
parser.add_argument(
    "--method_config",
    default=None,
    help="method config file",
)
parser.add_argument(
    "-a",
    "--arch",
    metavar="ARCH",
    default="resnet50",
    choices=timm.list_models(),
    help="model architecture: " + " | ".join(timm.list_models()) + " (default: resnet50)",
)
parser.add_argument(
    "--dataset", type=str,  default="cifar10", help="which dataset is used"
)
parser.add_argument(
    "--dataset_path", type=str,  default=None, help="Where is the dataset"
)
parser.add_argument(
    "--cache_path", type=str,  default=None, help="Where is the dataset cache"
)
parser.add_argument(
    "--method", type=str,  default="SimCLR", help="which method is used"
)
parser.add_argument(
    "-j",
    "--workers",
    default=16,
    type=int,
    metavar="N",
    help="number of data loading workers (default: 16)",
)
parser.add_argument(
    "-v",
    "--val-workers",
    default=16,
    type=int,
    metavar="N",
    help="number of val data loading workers (default: 16)",
)
parser.add_argument(
    "--epochs", default=100, type=int, metavar="N", help="number of total epochs to run"
)
parser.add_argument(
    "--start-epoch",
    default=0,
    type=int,
    metavar="N",
    help="manual epoch number (useful on restarts)",
)
parser.add_argument(
    "-b",
    "--batch-size",
    default=256,
    type=int,
    metavar="N",
    help="mini-batch size (default: 256), this is the total "
    "batch size of all GPUs on the current node when "
    "using Data Parallel or Distributed Data Parallel",
)
parser.add_argument(
    "--size",
    default=128,
    type=int,
    help="image size (default: 128)",
)
parser.add_argument(
    "--lr",
    "--learning-rate",
    default=0.1,
    type=float,
    metavar="LR",
    help="initial learning rate",
    dest="lr",
)
parser.add_argument(
    "--optimizer",
    default="AdamW",
    type=str,
    help="optimizer",
)
parser.add_argument(
    "--schedule",
    default=[60, 80],
    nargs="*",
    type=int,
    help="learning rate schedule (when to drop lr by a ratio)",
)
parser.add_argument(
    "--momentum", default=0.9, type=float, metavar="M", help="momentum"
)
parser.add_argument(
    "--wd",
    "--weight-decay",
    default=0.0001,
    type=float,
    metavar="W",
    help="weight decay (default: 0.)",
    dest="weight_decay",
)
parser.add_argument(
    "--save-path",
    default='.',
    type=str,
    help="weight save path",
)
parser.add_argument(
    "-p",
    "--print-freq",
    default=10,
    type=int,
    metavar="N",
    help="print frequency (default: 10)",
)
parser.add_argument(
    "--resume",
    default="",
    type=str,
    metavar="PATH",
    help="path to latest checkpoint (default: none)",
)
parser.add_argument(
    "-e",
    "--evaluate",
    dest="evaluate",
    action="store_true",
    help="evaluate model on validation set",
)
parser.add_argument(
    "--world-size",
    default=-1,
    type=int,
    help="number of nodes for distributed training",
)
parser.add_argument(
    "--rank", default=-1, type=int, help="node rank for distributed training"
)
parser.add_argument(
    "--dist-url",
    default="tcp://224.66.41.62:23456",
    type=str,
    help="url used to set up distributed training",
)
parser.add_argument(
    "--dist-backend", default="nccl", type=str, help="distributed backend"
)
parser.add_argument(
    "--seed", default=None, type=int, help="seed for initializing training. "
)
parser.add_argument(
    "--gpu", default=None, type=int, help="GPU id to use."
)
parser.add_argument(
    "--multiprocessing-distributed",
    action="store_true",
    help="Use multi-processing distributed training to launch "
    "N processes per node, which has N GPUs. This is the "
    "fastest way to use PyTorch for either single node or "
    "multi node data parallel training",
)
parser.add_argument(
    "--num-classes", type=int,  default=10, help="which method is used"
)
parser.add_argument(
    "--pretrained", default="", type=str, help="path to moco pretrained checkpoint"
)
parser.add_argument(
    "--suffix", type=str,  default="", help="which method is used"
)

#used for SNN
parser.add_argument(
    "--step", type=int,  default=4, help="which method is used"
)
parser.add_argument(
    "--layer_by_layer", action="store_true", help="Is layer by layer used"
)
# Used for moco
parser.add_argument(
    "--output-dim", default=128, type=int, help="feature dimension (default: 128)"
)
parser.add_argument(
    "--moco-k",
    default=65536,
    type=int,
    help="queue size; number of negative keys (default: 65536)",
)
parser.add_argument(
    "--moco-m",
    default=0.999,
    type=float,
    help="moco momentum of updating key encoder (default: 0.999)",
)
parser.add_argument(
    "--moco-t", default=0.07, type=float, help="softmax temperature (default: 0.07)"
)

# options for moco v2
parser.add_argument(
    "--mlp", action="store_true", help="use mlp head"
)
parser.add_argument(
    "--aug-plus", action="store_true", help="use moco v2 data augmentation"
)
parser.add_argument(
    "--cos", action="store_true", help="use cosine lr schedule"
)

# options for SimSiam or PredNext
parser.add_argument(
    "--pred-dim", type=int,  default=512, help="which method is used"
)

 
parser.add_argument(
    "--frames-per-clip", type=int,  default=10, help="frames per clip (default: 10)"
)
parser.add_argument(
    "--frame-rate", type=float,  default=1, help="frame rate (default: 25/8)"
)
parser.add_argument(
    "--knn-k", type=int,  default=200, help="knn k (default: 200)"
)
parser.add_argument(
    "--knn-t", type=float,  default=0.1, help="knn t (default: 0.1)"
)
parser.add_argument(
    "--knn-freq", type=int,  default=1, help="knn freq (default: 10)"
)
parser.add_argument(
    "--warmup-epochs", type=int,  default=0, help="warmup epochs (default: 10)"
)
parser.add_argument(
    "--enable-knn", action="store_true", help="Is knn used"
)
parser.add_argument(
    "--knn-max-samples", type=int,  default=99999999, help="knn max samples (default: 99999999)"
)
parser.add_argument(
    "--sync-bn", action="store_true", help="Is sync bn used"
)
parser.add_argument(
    "--node-type", type=str,  default="LIFNode", help="node type (default: LIFNode)"
)
parser.add_argument(
    "--zero-init-residual", action="store_true", help="Is zero init residual used"
)
parser.add_argument(
    "--channel-first", action="store_true", help="Is channel first used"
)
parser.add_argument(
    "--num-clips", type=int,  default=1, help="number of clips (default: 1)"
)
parser.add_argument(
    "--fc-lr-scale", type=float,  default=1.0, help="fc lr scale (default: 1.0)"
)
parser.add_argument(
    "--lr-scale-epochs", type=int,  default=0, help="lr scale epochs (default: 0)"
)
best_acc1 = 0


def main():
    args = parser.parse_args()
    args = load_config(args)
 
        
    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn(
            "You have chosen to seed training. "
            "This will turn on the CUDNN deterministic setting, "
            "which can slow down your training considerably! "
            "You may see unexpected behavior when restarting "
            "from checkpoints."
        )

    if args.gpu is not None:
        warnings.warn(
            "You have chosen a specific GPU. This will completely "
            "disable data parallelism."
        )

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    args.ngpus_per_node = ngpus_per_node
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args):
    global best_acc1
    args.gpu = gpu

    # suppress printing if not master
    if args.multiprocessing_distributed and args.gpu != 0:

        def print_pass(*args):
            pass

        builtins.print = print_pass

    if args.gpu is not None:
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(
            backend=args.dist_backend,
            init_method=args.dist_url,
            world_size=args.world_size,
            rank=args.rank,
        )
    # create model
    print("=> creating model '{}'".format(args.arch))
    
    
    if is_SNN(args.arch):
        model = timm.models.model_entrypoint(args.arch)(
            dataset=args.dataset,
            layer_by_layer=args.layer_by_layer,
            is_cifar=True if "cifar" in args.dataset else False,
            step=args.step,
            num_classes=args.num_classes,
            node_type=eval(args.node_type),
            zero_init_residual=args.zero_init_residual,
            )
    else:
        model = timm.models.model_entrypoint(args.arch)(num_classes=args.num_classes)
    exchange_conv(model,args)



    # load from pre-trained, before DistributedDataParallel constructor
    if args.pretrained:
        if os.path.isfile(args.pretrained):
            print("=> loading pretrained checkpoint '{}'".format(args.pretrained))
            checkpoint = torch.load(args.pretrained, map_location="cpu", weights_only=False)
            try:
                state_dict = checkpoint["state_dict"]
            except:
                state_dict = checkpoint.get("model", checkpoint)
            # remove leading 'module.' if present (loading into non-DDP model)
            if all(k.startswith("module.") for k in state_dict.keys()):
                state_dict = {k[len("module."):]: v for k, v in state_dict.items()}
            state_dict = {'.'.join(p for p in k.split('.') if p != 'module'): v for k, v in state_dict.items()}    
            state_dict = {'.'.join(p for p in k.split('.') if p != 'encoder_q'): v for k, v in state_dict.items()}    
            state_dict = {k: v for k, v in state_dict.items() if "fc" not in k}
            msg = model.load_state_dict(state_dict, strict=False)
            try:
                missing = list(msg.missing_keys)
                missing = [k for k in missing if "alpha" not in k and "threshold" not in k]
                unexpected = list(msg.unexpected_keys)
                if missing or unexpected:
                    print(f"=> loaded with missing keys: {missing} \n\n unexpected keys: {unexpected}")
            except Exception:
                pass
            print("=> pretrained weights loaded")
        else:
            print("=> no pretrained checkpoint found at '{}'".format(args.pretrained))

    if args.distributed and args.sync_bn:
        print("=> using SyncBatchNorm")
        model = torch.nn.SyncBatchNorm.convert_sync_batchnorm(model)
    print(model)
    args.init_lr = args.lr * args.batch_size / 256
    if args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            args.val_workers = int((args.val_workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(
                model, device_ids=[args.gpu]
            )
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
    else:
        # DataParallel will divide and allocate batch_size to all available GPUs
        if args.arch.startswith("alexnet") or args.arch.startswith("vgg"):
            model.features = torch.nn.DataParallel(model.features)
            model.cuda()
        else:
            model = torch.nn.DataParallel(model).cuda()

 

    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss().cuda(args.gpu)
    scaler = GradScaler()
    # optimize all parameters
    parameters = list(filter(lambda p: p.requires_grad, model.parameters()))
    parameters = [
        {'params': parameters[:-2], 'name': 'backbone'},
        {'params': parameters[-2:], 'name': 'fc'}
    ]
    if args.optimizer == "SGD":
        optimizer = torch.optim.SGD(
            parameters, args.init_lr, momentum=args.momentum, weight_decay=args.weight_decay
        )
    elif args.optimizer == "AdamW":
        optimizer = torch.optim.AdamW(
            parameters, args.init_lr, weight_decay=args.weight_decay
        )
    else:
        optimizer = eval(f"torch.optim.{args.optimizer}")(
            parameters, args.init_lr, weight_decay=args.weight_decay
        )

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume,weights_only=False)
            else:
                # Map model to be loaded to specified single gpu.
                loc = "cuda:{}".format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc,weights_only=False)
            args.start_epoch = checkpoint["epoch"]
            best_acc1 = checkpoint["best_acc1"]
            if args.gpu is not None:
                # best_acc1 may be from a checkpoint from a different GPU
                best_acc1 = best_acc1.to(args.gpu)
            model.load_state_dict(checkpoint["state_dict"])
            optimizer.load_state_dict(checkpoint["optimizer"])
            print(
                "=> loaded checkpoint '{}' (epoch {})".format(
                    args.resume, checkpoint["epoch"]
                )
            )
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    print("begin to load dataset")
    train_dataset,test_dataset = eval(f"get_{args.dataset}_datasets")(args,linearCls=True)
    print("load dataset finished")
    
    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None

    train_loader = torch.utils.data.DataLoader(
        train_dataset,
        batch_size=args.batch_size,
        shuffle=(train_sampler is None),
        num_workers=args.workers,
        pin_memory=True,
        sampler=train_sampler,
        persistent_workers=True,
        prefetch_factor=3
    )

    val_loader = torch.utils.data.DataLoader(
        test_dataset,
        batch_size=args.batch_size,
        shuffle=False,
        num_workers=args.val_workers,
        pin_memory=True,
        persistent_workers=True,
        prefetch_factor=1
    )
    if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % ngpus_per_node == 0
        ):
        wandb.init(
            project="finetune_CrossTP",
            config=args,
        
        )
     

    if args.evaluate:
        validate(val_loader, model, criterion, 0, optimizer, args)
        return
 


    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            train_sampler.set_epoch(epoch)
        args.fc_lr_scale = args.fc_lr_scale if epoch < args.lr_scale_epochs else 1.0
        adjust_learning_rate(optimizer, epoch, args)

        # train for one epoch
        train(train_loader, model, criterion, optimizer, scaler, epoch, args)

        # evaluate on validation set
        acc1 = validate(val_loader, model, criterion, epoch, optimizer, args)

        if epoch % args.knn_freq == 0 and args.enable_knn:
            backup_model_fc = model.module.fc
            model.module.fc = nn.Identity()
            backup_model_sum_output = model.module.sum_output
            model.module.sum_output = False
            knn_acc1, knn_acc5, knn_std, knn_temporal_feature_similarity = knn_test(model, train_loader, val_loader, args)
            model.module.fc = backup_model_fc
            model.module.sum_output = backup_model_sum_output

        # remember best acc@1 and save checkpoint
        is_best = acc1 > best_acc1
        best_acc1 = max(acc1, best_acc1)

        if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % ngpus_per_node == 0
        ):
            if is_best:
                pretrained_filename=args.pretrained.split("/")[-1][:-8]
                try:
                    save_path=os.path.join(args.save_path,f"./checkpoints/finetune/{args.dataset}/{args.method}/{args.arch}/{pretrained_filename}-{wandb.run.id}{args.suffix}-{epoch+1}.pth.tar")
                except:
                    save_path=os.path.join(args.save_path,f"./checkpoints/finetune/{args.dataset}/{args.method}/{args.arch}/{pretrained_filename}-{wandb.get_run().public.run_id}{args.suffix}-{epoch+1}.pth.tar")
                save_checkpoint(
                    {
                        "epoch": epoch + 1,
                        "arch": args.arch,
                        "state_dict": model.state_dict(),
                        "acc1": acc1,
                        "optimizer": optimizer.state_dict(),
                    },
                    filename=save_path
                )
    if args.distributed:
        dist.destroy_process_group()


def train(train_loader, model, criterion, optimizer, scaler, epoch, args):
    batch_time = AverageMeter("Time", ":6.3f")
    data_time = AverageMeter("Data", ":6.3f")
    losses = AverageMeter("Loss", ":.4e")
    top1 = AverageMeter("Acc@1", ":6.2f")
    top5 = AverageMeter("Acc@5", ":6.2f")
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, losses, top1, top5],
        prefix="Epoch: [{}]".format(epoch),
    )

    """
    Switch to train mode for fine-tuning:
    Unlike linear classification, fine-tuning requires the model to be in train mode
    so that BatchNorm and Dropout layers work properly during training.
    """
    model.train()

    end = time.time()
    for i, (images, target) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)
        if args.gpu is not None:
            images = images.cuda(args.gpu, non_blocking=True)
        target = target.cuda(args.gpu, non_blocking=True)
        # compute output
        with autocast(device_type=images.device.type):
            output = model(images)
            loss = criterion(output, target)
        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        losses.update(loss.item(), images.size(0))
        top1.update(acc1[0], images.size(0))
        top5.update(acc5[0], images.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()

        # loss.backward()
        # torch.nn.utils.clip_grad_norm_(model.parameters(), 1.0)
        # optimizer.step()
        
        scaler.scale(loss).backward()
        # Unscale gradients before clipping when using AMP
        # scaler.unscale_(optimizer)
        # torch.nn.utils.clip_grad_norm_(model.parameters(), 1.0)
        scaler.step(optimizer)
        scaler.update()
        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()
        dist.barrier()

        if i % args.print_freq == 0:
            progress.display(i)

def validate(val_loader, model, criterion, epoch, optimizer, args):
    batch_time = AverageMeter("Time", ":6.3f")
    losses = AverageMeter("Loss", ":.4e")
    top1 = AverageMeter("Acc@1", ":6.2f")
    top5 = AverageMeter("Acc@5", ":6.2f")
    progress = ProgressMeter(
        len(val_loader), [batch_time, losses, top1, top5], prefix="Test: "
    )

    # switch to evaluate mode
    model.eval()

    with torch.no_grad():
        end = time.time()
        for i, (images, target) in enumerate(val_loader):
            if args.gpu is not None:
                images = images.cuda(args.gpu, non_blocking=True)
            target = target.cuda(args.gpu, non_blocking=True)

            # compute output
            if len(images.shape) == 6:
                output = model(images.view(-1, *images.shape[2:]))
                output = output.view(images.shape[0], -1, output.shape[-1]).mean(dim=1)
            else:
                output = model(images)
            loss = criterion(output, target)

            # measure accuracy and record loss
            acc1, acc5 = accuracy(output, target, topk=(1, 5))
            losses.update(loss.item(), images.size(0))
            top1.update(acc1[0], images.size(0))
            top5.update(acc5[0], images.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()
            if i % args.print_freq == 0:
                progress.display(i)
            dist.barrier()
        print(
            " * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}".format(top1=top1, top5=top5)
        )
        
        if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % args.ngpus_per_node == 0
        ):            
            wandb.log({
                "epoch": epoch,
                "lr": optimizer.param_groups[0]['lr'],
                "losses": losses.avg,
                "top1": top1.avg,
                "top5": top5.avg,
            })
    return top1.avg


 

if __name__ == "__main__":
    main()
